<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('admissions', function (Blueprint $table) {
            // Check if columns don't already exist before adding
            if (!Schema::hasColumn('admissions', 'admission_number')) {
                $table->string('admission_number')->unique()->after('id');
            }
            if (!Schema::hasColumn('admissions', 'patient_id')) {
                $table->foreignId('patient_id')->constrained()->onDelete('cascade')->after('admission_number');
            }
            if (!Schema::hasColumn('admissions', 'visit_id')) {
                $table->foreignId('visit_id')->nullable()->constrained()->onDelete('set null')->after('patient_id');
            }
            if (!Schema::hasColumn('admissions', 'admitted_by')) {
                $table->foreignId('admitted_by')->constrained('users')->onDelete('cascade')->after('visit_id');
            }
            if (!Schema::hasColumn('admissions', 'discharged_by')) {
                $table->foreignId('discharged_by')->nullable()->constrained('users')->onDelete('set null')->after('admitted_by');
            }
            if (!Schema::hasColumn('admissions', 'ward')) {
                $table->string('ward')->nullable()->after('discharged_by');
            }
            if (!Schema::hasColumn('admissions', 'room_number')) {
                $table->string('room_number')->nullable()->after('ward');
            }
            if (!Schema::hasColumn('admissions', 'bed_number')) {
                $table->string('bed_number')->nullable()->after('room_number');
            }
            if (!Schema::hasColumn('admissions', 'admission_reason')) {
                $table->text('admission_reason')->after('bed_number');
            }
            if (!Schema::hasColumn('admissions', 'diagnosis')) {
                $table->text('diagnosis')->nullable()->after('admission_reason');
            }
            if (!Schema::hasColumn('admissions', 'admission_notes')) {
                $table->text('admission_notes')->nullable()->after('diagnosis');
            }
            if (!Schema::hasColumn('admissions', 'status')) {
                // For PostgreSQL, use string with check constraint
                if (DB::getDriverName() === 'pgsql') {
                    $table->string('status')->default('admitted')->after('admission_notes');
                } else {
                    $table->enum('status', ['admitted', 'discharged', 'transferred'])->default('admitted')->after('admission_notes');
                }
            }
            if (!Schema::hasColumn('admissions', 'admission_date')) {
                $table->timestamp('admission_date')->after('status');
            }
            if (!Schema::hasColumn('admissions', 'discharge_date')) {
                $table->timestamp('discharge_date')->nullable()->after('admission_date');
            }
            if (!Schema::hasColumn('admissions', 'discharge_notes')) {
                $table->text('discharge_notes')->nullable()->after('discharge_date');
            }
            if (!Schema::hasColumn('admissions', 'discharge_instructions')) {
                $table->text('discharge_instructions')->nullable()->after('discharge_notes');
            }
        });

        // Add check constraint for status in PostgreSQL after column is created
        if (DB::getDriverName() === 'pgsql' && Schema::hasColumn('admissions', 'status')) {
            try {
                DB::statement("ALTER TABLE admissions ADD CONSTRAINT admissions_status_check CHECK (status IN ('admitted', 'discharged', 'transferred'))");
            } catch (\Exception $e) {
                // Constraint might already exist
            }
        }

        // Add indexes (will fail gracefully if they already exist)
        try {
            Schema::table('admissions', function (Blueprint $table) {
                $table->index('patient_id');
                $table->index('visit_id');
                $table->index('status');
                $table->index('admission_date');
            });
        } catch (\Exception $e) {
            // Indexes might already exist, continue
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('admissions', function (Blueprint $table) {
            $table->dropForeign(['patient_id']);
            $table->dropForeign(['visit_id']);
            $table->dropForeign(['admitted_by']);
            $table->dropForeign(['discharged_by']);
            $table->dropColumn([
                'admission_number',
                'patient_id',
                'visit_id',
                'admitted_by',
                'discharged_by',
                'ward',
                'room_number',
                'bed_number',
                'admission_reason',
                'diagnosis',
                'admission_notes',
                'status',
                'admission_date',
                'discharge_date',
                'discharge_notes',
                'discharge_instructions',
            ]);
        });

        // Drop enum type for PostgreSQL
        if (DB::getDriverName() === 'pgsql') {
            DB::statement('DROP TYPE IF EXISTS admissions_status_enum');
        }
    }
};
