<?php

namespace App\Services;

use App\Models\AuditLog;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;

class AuditLogService
{
    /**
     * Log a change to a model field
     */
    public static function logChange(
        Model $model,
        string $fieldName,
        $oldValue,
        $newValue,
        ?int $patientId = null,
        ?int $visitId = null,
        ?string $description = null
    ): AuditLog {
        // Determine patient ID if not provided
        if (!$patientId && method_exists($model, 'patient_id')) {
            $patientId = $model->patient_id;
        } elseif (!$patientId && $model instanceof \App\Models\Patient) {
            $patientId = $model->id;
        }

        // Determine action
        $action = $model->wasRecentlyCreated ? 'created' : 'updated';

        // Create description if not provided
        if (!$description) {
            $description = self::generateDescription($fieldName, $oldValue, $newValue, $action);
        }

        return AuditLog::create([
            'user_id' => Auth::id(),
            'patient_id' => $patientId,
            'visit_id' => $visitId,
            'model_type' => get_class($model),
            'model_id' => $model->id,
            'action' => $action,
            'field_name' => $fieldName,
            'old_value' => $oldValue !== null ? (string) $oldValue : null,
            'new_value' => $newValue !== null ? (string) $newValue : null,
            'description' => $description,
            'ip_address' => Request::ip(),
            'user_agent' => Request::userAgent(),
        ]);
    }

    /**
     * Log medical history update
     */
    public static function logMedicalHistoryUpdate(
        \App\Models\Patient $patient,
        ?string $oldValue,
        ?string $newValue,
        ?int $visitId = null
    ): AuditLog {
        return self::logChange(
            $patient,
            'medical_history',
            $oldValue,
            $newValue,
            $patient->id,
            $visitId,
            'Medical history updated'
        );
    }

    /**
     * Log allergies update
     */
    public static function logAllergiesUpdate(
        \App\Models\Patient $patient,
        ?string $oldValue,
        ?string $newValue,
        ?int $visitId = null
    ): AuditLog {
        return self::logChange(
            $patient,
            'allergies',
            $oldValue,
            $newValue,
            $patient->id,
            $visitId,
            'Allergies updated'
        );
    }

    /**
     * Generate human-readable description
     */
    protected static function generateDescription(
        string $fieldName,
        $oldValue,
        $newValue,
        string $action
    ): string {
        $fieldLabel = match($fieldName) {
            'medical_history' => 'Medical History',
            'allergies' => 'Allergies',
            'height' => 'Height',
            'weight' => 'Weight',
            default => ucfirst(str_replace('_', ' ', $fieldName)),
        };

        if ($action === 'created') {
            return "{$fieldLabel} created";
        }

        if (empty($oldValue) && !empty($newValue)) {
            return "{$fieldLabel} added";
        }

        if (!empty($oldValue) && empty($newValue)) {
            return "{$fieldLabel} removed";
        }

        return "{$fieldLabel} updated";
    }

    /**
     * Get audit logs for a patient
     */
    public static function getPatientLogs(int $patientId, ?string $fieldName = null)
    {
        $query = AuditLog::forPatient($patientId)
            ->with(['user', 'visit'])
            ->orderBy('created_at', 'desc');

        if ($fieldName) {
            $query->forField($fieldName);
        }

        return $query->get();
    }

    /**
     * Get audit logs for a visit
     */
    public static function getVisitLogs(int $visitId)
    {
        return AuditLog::forVisit($visitId)
            ->with(['user', 'patient'])
            ->orderBy('created_at', 'desc')
            ->get();
    }
}


