<?php

namespace App\Services;

use App\Models\Drug;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;

class AIDrugInteractionService
{
    private $apiKey;
    private $baseUrl = 'https://api.openai.com/v1/chat/completions';

    public function __construct()
    {
        $this->apiKey = config('services.openai.api_key');
    }

    /**
     * Check for drug interactions using AI
     */
    public function checkInteractions(array $drugIds): array
    {
        $drugs = Drug::whereIn('id', $drugIds)->get();
        
        if ($drugs->count() < 2) {
            return ['interactions' => [], 'risk_level' => 'none'];
        }

        $cacheKey = 'drug_interactions_' . md5(implode(',', $drugIds));
        
        return Cache::remember($cacheKey, 3600, function () use ($drugs) {
            return $this->analyzeWithAI($drugs);
        });
    }

    /**
     * Analyze drug interactions using OpenAI
     */
    private function analyzeWithAI($drugs): array
    {
        $drugInfo = $drugs->map(function ($drug) {
            return [
                'name' => $drug->name,
                'generic_name' => $drug->generic_name,
                'category' => $drug->category,
                'contraindications' => $drug->contraindications,
                'interactions' => $drug->interactions,
            ];
        })->toArray();

        $prompt = $this->buildInteractionPrompt($drugInfo);

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(30)->post($this->baseUrl, [
                'model' => 'gpt-3.5-turbo',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a medical AI assistant specializing in drug interactions. Provide accurate, evidence-based analysis of potential drug interactions. Always prioritize patient safety.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 1000,
                'temperature' => 0.3,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return $this->parseAIResponse($result['choices'][0]['message']['content']);
            }
        } catch (\Exception $e) {
            \Log::error('AI Drug Interaction Check Failed: ' . $e->getMessage());
        }

        return $this->fallbackInteractionCheck($drugs);
    }

    /**
     * Build the prompt for AI analysis
     */
    private function buildInteractionPrompt(array $drugs): string
    {
        $drugList = collect($drugs)->map(function ($drug) {
            return "- {$drug['name']} ({$drug['generic_name']}) - Category: {$drug['category']}";
        })->join("\n");

        return "Analyze the following drugs for potential interactions:\n\n{$drugList}\n\n" .
               "Please provide:\n" .
               "1. Risk level (none, low, moderate, high, severe)\n" .
               "2. Specific interactions found\n" .
               "3. Clinical significance\n" .
               "4. Recommendations\n\n" .
               "Format your response as JSON with keys: risk_level, interactions, clinical_significance, recommendations";
    }

    /**
     * Parse AI response into structured data
     */
    private function parseAIResponse(string $response): array
    {
        try {
            // Try to extract JSON from response
            if (preg_match('/\{.*\}/s', $response, $matches)) {
                $jsonData = json_decode($matches[0], true);
                if ($jsonData) {
                    return [
                        'risk_level' => $jsonData['risk_level'] ?? 'unknown',
                        'interactions' => $jsonData['interactions'] ?? [],
                        'clinical_significance' => $jsonData['clinical_significance'] ?? '',
                        'recommendations' => $jsonData['recommendations'] ?? '',
                        'ai_generated' => true,
                    ];
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to parse AI response: ' . $e->getMessage());
        }

        return [
            'risk_level' => 'unknown',
            'interactions' => [],
            'clinical_significance' => 'AI analysis failed',
            'recommendations' => 'Please consult with a pharmacist',
            'ai_generated' => false,
        ];
    }

    /**
     * Fallback interaction check using local data
     */
    private function fallbackInteractionCheck($drugs): array
    {
        $interactions = [];
        $riskLevel = 'low';

        foreach ($drugs as $drug) {
            if ($drug->interactions) {
                foreach ($drug->interactions as $interaction) {
                    $interactions[] = [
                        'drug' => $drug->name,
                        'interaction' => $interaction,
                        'severity' => 'moderate',
                    ];
                }
            }
        }

        if (!empty($interactions)) {
            $riskLevel = 'moderate';
        }

        return [
            'risk_level' => $riskLevel,
            'interactions' => $interactions,
            'clinical_significance' => 'Local database check',
            'recommendations' => 'Consider consulting with a pharmacist for detailed analysis',
            'ai_generated' => false,
        ];
    }

    /**
     * Get drug recommendation based on symptoms
     */
    public function getDrugRecommendations(string $symptoms, array $patientAllergies = []): array
    {
        $cacheKey = 'drug_recommendations_' . md5($symptoms . implode(',', $patientAllergies));
        
        return Cache::remember($cacheKey, 1800, function () use ($symptoms, $patientAllergies) {
            return $this->getAIRecommendations($symptoms, $patientAllergies);
        });
    }

    /**
     * Get AI-powered drug recommendations
     */
    private function getAIRecommendations(string $symptoms, array $patientAllergies): array
    {
        $prompt = "Based on the following symptoms: '{$symptoms}', suggest appropriate medications. ";
        
        if (!empty($patientAllergies)) {
            $prompt .= "Patient has allergies to: " . implode(', ', $patientAllergies) . ". ";
        }
        
        $prompt .= "Provide 3-5 medication recommendations with dosages and considerations. " .
                  "Format as JSON with keys: recommendations (array of objects with name, dosage, frequency, considerations)";

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(30)->post($this->baseUrl, [
                'model' => 'gpt-3.5-turbo',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a medical AI assistant. Provide evidence-based medication recommendations. Always consider patient safety and contraindications.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 800,
                'temperature' => 0.3,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return $this->parseRecommendationsResponse($result['choices'][0]['message']['content']);
            }
        } catch (\Exception $e) {
            \Log::error('AI Drug Recommendation Failed: ' . $e->getMessage());
        }

        return ['recommendations' => [], 'ai_generated' => false];
    }

    /**
     * Parse AI recommendations response
     */
    private function parseRecommendationsResponse(string $response): array
    {
        try {
            if (preg_match('/\{.*\}/s', $response, $matches)) {
                $jsonData = json_decode($matches[0], true);
                if ($jsonData) {
                    return [
                        'recommendations' => $jsonData['recommendations'] ?? [],
                        'ai_generated' => true,
                    ];
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to parse AI recommendations: ' . $e->getMessage());
        }

        return ['recommendations' => [], 'ai_generated' => false];
    }
}