<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;

class AIDiagnosisService
{
    private $apiKey;
    private $baseUrl = 'https://api.openai.com/v1/chat/completions';

    public function __construct()
    {
        $this->apiKey = config('services.openai.api_key');
    }

    /**
     * Get AI-powered diagnosis suggestions
     */
    public function getDiagnosisSuggestions(array $symptoms, array $vitalSigns = [], array $patientHistory = []): array
    {
        $cacheKey = 'diagnosis_' . md5(serialize($symptoms) . serialize($vitalSigns) . serialize($patientHistory));
        
        return Cache::remember($cacheKey, 1800, function () use ($symptoms, $vitalSigns, $patientHistory) {
            return $this->analyzeSymptomsWithAI($symptoms, $vitalSigns, $patientHistory);
        });
    }

    /**
     * Analyze symptoms using AI
     */
    private function analyzeSymptomsWithAI(array $symptoms, array $vitalSigns, array $patientHistory): array
    {
        $prompt = $this->buildDiagnosisPrompt($symptoms, $vitalSigns, $patientHistory);

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(30)->post($this->baseUrl, [
                'model' => 'gpt-4',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a medical AI assistant specializing in differential diagnosis. Provide evidence-based diagnostic suggestions based on symptoms, vital signs, and patient history. Always emphasize that this is for assistance only and final diagnosis should be made by qualified healthcare professionals.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 1200,
                'temperature' => 0.2,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return $this->parseDiagnosisResponse($result['choices'][0]['message']['content']);
            }
        } catch (\Exception $e) {
            \Log::error('AI Diagnosis Analysis Failed: ' . $e->getMessage());
        }

        return $this->getFallbackDiagnosis($symptoms);
    }

    /**
     * Build diagnosis prompt
     */
    private function buildDiagnosisPrompt(array $symptoms, array $vitalSigns, array $patientHistory): string
    {
        $prompt = "Based on the following clinical information, provide diagnostic suggestions:\n\n";
        
        $prompt .= "SYMPTOMS:\n";
        foreach ($symptoms as $symptom) {
            $prompt .= "- {$symptom}\n";
        }

        if (!empty($vitalSigns)) {
            $prompt .= "\nVITAL SIGNS:\n";
            foreach ($vitalSigns as $key => $value) {
                $prompt .= "- {$key}: {$value}\n";
            }
        }

        if (!empty($patientHistory)) {
            $prompt .= "\nRELEVANT PATIENT HISTORY:\n";
            foreach ($patientHistory as $history) {
                $prompt .= "- {$history}\n";
            }
        }

        $prompt .= "\nPlease provide:\n";
        $prompt .= "1. Top 3-5 differential diagnoses (most likely first)\n";
        $prompt .= "2. Confidence level for each diagnosis (low/medium/high)\n";
        $prompt .= "3. Recommended diagnostic tests\n";
        $prompt .= "4. Red flag symptoms to watch for\n";
        $prompt .= "5. Urgency level (low/medium/high/emergency)\n\n";
        $prompt .= "Format as JSON with keys: diagnoses, diagnostic_tests, red_flags, urgency_level, confidence_notes";

        return $prompt;
    }

    /**
     * Parse AI diagnosis response
     */
    private function parseDiagnosisResponse(string $response): array
    {
        try {
            if (preg_match('/\{.*\}/s', $response, $matches)) {
                $jsonData = json_decode($matches[0], true);
                if ($jsonData) {
                    return [
                        'diagnoses' => $jsonData['diagnoses'] ?? [],
                        'diagnostic_tests' => $jsonData['diagnostic_tests'] ?? [],
                        'red_flags' => $jsonData['red_flags'] ?? [],
                        'urgency_level' => $jsonData['urgency_level'] ?? 'medium',
                        'confidence_notes' => $jsonData['confidence_notes'] ?? '',
                        'ai_generated' => true,
                    ];
    }
}
        } catch (\Exception $e) {
            \Log::error('Failed to parse AI diagnosis response: ' . $e->getMessage());
        }

        return $this->getFallbackDiagnosis([]);
    }

    /**
     * Fallback diagnosis suggestions
     */
    private function getFallbackDiagnosis(array $symptoms): array
    {
        return [
            'diagnoses' => [
                [
                    'condition' => 'General consultation needed',
                    'confidence' => 'medium',
                    'description' => 'Further clinical assessment required'
                ]
            ],
            'diagnostic_tests' => ['Complete blood count', 'Basic metabolic panel'],
            'red_flags' => ['Severe pain', 'High fever', 'Difficulty breathing'],
            'urgency_level' => 'medium',
            'confidence_notes' => 'AI analysis unavailable - manual assessment recommended',
            'ai_generated' => false,
        ];
    }

    /**
     * Analyze treatment response
     */
    public function analyzeTreatmentResponse(array $symptoms, array $treatment, int $daysSinceTreatment): array
    {
        $cacheKey = 'treatment_response_' . md5(serialize($symptoms) . serialize($treatment) . $daysSinceTreatment);
        
        return Cache::remember($cacheKey, 3600, function () use ($symptoms, $treatment, $daysSinceTreatment) {
            return $this->analyzeTreatmentWithAI($symptoms, $treatment, $daysSinceTreatment);
        });
    }

    /**
     * Analyze treatment effectiveness with AI
     */
    private function analyzeTreatmentWithAI(array $symptoms, array $treatment, int $daysSinceTreatment): array
    {
        $prompt = "Analyze the effectiveness of the following treatment:\n\n";
        $prompt .= "INITIAL SYMPTOMS:\n";
        foreach ($symptoms as $symptom) {
            $prompt .= "- {$symptom}\n";
        }

        $prompt .= "\nTREATMENT GIVEN:\n";
        foreach ($treatment as $med) {
            $prompt .= "- {$med}\n";
        }

        $prompt .= "\nDAYS SINCE TREATMENT STARTED: {$daysSinceTreatment}\n\n";
        $prompt .= "Provide analysis including:\n";
        $prompt .= "1. Expected response timeline\n";
        $prompt .= "2. Signs of improvement to look for\n";
        $prompt .= "3. Signs of treatment failure\n";
        $prompt .= "4. Recommendations for follow-up\n";
        $prompt .= "5. When to consider treatment modification\n\n";
        $prompt .= "Format as JSON with keys: expected_timeline, improvement_signs, failure_signs, follow_up_recommendations, modification_triggers";

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(30)->post($this->baseUrl, [
                'model' => 'gpt-3.5-turbo',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a medical AI assistant analyzing treatment effectiveness. Provide evidence-based analysis of expected treatment responses and monitoring recommendations.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 1000,
                'temperature' => 0.3,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return $this->parseTreatmentResponse($result['choices'][0]['message']['content']);
            }
        } catch (\Exception $e) {
            \Log::error('AI Treatment Analysis Failed: ' . $e->getMessage());
        }

        return [
            'expected_timeline' => 'Varies by condition',
            'improvement_signs' => ['Symptom reduction', 'Improved vital signs'],
            'failure_signs' => ['Worsening symptoms', 'No improvement'],
            'follow_up_recommendations' => ['Monitor symptoms', 'Follow up in 1 week'],
            'modification_triggers' => ['No improvement after 3-5 days'],
            'ai_generated' => false,
        ];
    }

    /**
     * Parse treatment response
     */
    private function parseTreatmentResponse(string $response): array
    {
        try {
            if (preg_match('/\{.*\}/s', $response, $matches)) {
                $jsonData = json_decode($matches[0], true);
                if ($jsonData) {
                    return array_merge($jsonData, ['ai_generated' => true]);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to parse AI treatment response: ' . $e->getMessage());
        }

        return [
            'expected_timeline' => 'Varies by condition',
            'improvement_signs' => ['Symptom reduction', 'Improved vital signs'],
            'failure_signs' => ['Worsening symptoms', 'No improvement'],
            'follow_up_recommendations' => ['Monitor symptoms', 'Follow up in 1 week'],
            'modification_triggers' => ['No improvement after 3-5 days'],
            'ai_generated' => false,
        ];
    }
}