<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;

class AIChatbotService
{
    private $apiKey;
    private $baseUrl = 'https://api.openai.com/v1/chat/completions';

    public function __construct()
    {
        $this->apiKey = config('services.openai.api_key');
    }

    /**
     * Get AI chatbot response
     */
    public function getResponse(string $message, array $context = []): array
    {
        $cacheKey = 'chatbot_response_' . md5($message . serialize($context));
        
        return Cache::remember($cacheKey, 300, function () use ($message, $context) {
            return $this->generateAIResponse($message, $context);
        });
    }

    /**
     * Generate AI response
     */
    private function generateAIResponse(string $message, array $context): array
    {
        $systemPrompt = $this->buildSystemPrompt($context);
        $userPrompt = $this->buildUserPrompt($message, $context);

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(15)->post($this->baseUrl, [
                'model' => 'gpt-3.5-turbo',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => $systemPrompt
                    ],
                    [
                        'role' => 'user',
                        'content' => $userPrompt
                    ]
                ],
                'max_tokens' => 500,
                'temperature' => 0.7,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return [
                    'response' => $result['choices'][0]['message']['content'],
                    'ai_generated' => true,
                    'confidence' => 'high'
                ];
            }
        } catch (\Exception $e) {
            \Log::error('AI Chatbot Response Failed: ' . $e->getMessage());
        }

        return $this->getFallbackResponse($message);
    }

    /**
     * Build system prompt for chatbot
     */
    private function buildSystemPrompt(array $context): string
    {
        $prompt = "You are a helpful AI assistant for a hospital's patient portal. ";
        $prompt .= "Your role is to assist patients with general health information, appointment scheduling, and basic medical questions. ";
        $prompt .= "IMPORTANT GUIDELINES:\n";
        $prompt .= "1. Always emphasize that you are an AI assistant and not a replacement for medical advice\n";
        $prompt .= "2. For urgent medical issues, direct patients to emergency services\n";
        $prompt .= "3. For specific medical advice, recommend consulting with healthcare providers\n";
        $prompt .= "4. Be empathetic, professional, and helpful\n";
        $prompt .= "5. Keep responses concise but informative\n";
        $prompt .= "6. If you don't know something, admit it and suggest contacting the hospital\n\n";

        if (!empty($context['patient_info'])) {
            $prompt .= "PATIENT CONTEXT:\n";
            $prompt .= "Patient ID: " . ($context['patient_info']['id'] ?? 'Unknown') . "\n";
            $prompt .= "Name: " . ($context['patient_info']['name'] ?? 'Unknown') . "\n";
        }

        if (!empty($context['appointments'])) {
            $prompt .= "UPCOMING APPOINTMENTS:\n";
            foreach ($context['appointments'] as $appointment) {
                $prompt .= "- " . $appointment['date'] . " with " . $appointment['doctor'] . "\n";
            }
        }

        return $prompt;
    }

    /**
     * Build user prompt
     */
    private function buildUserPrompt(string $message, array $context): string
    {
        $prompt = "Patient message: \"{$message}\"\n\n";
        
        if (!empty($context['recent_visits'])) {
            $prompt .= "Recent medical history context:\n";
            foreach ($context['recent_visits'] as $visit) {
                $prompt .= "- " . $visit['date'] . ": " . $visit['summary'] . "\n";
            }
        }

        return $prompt;
    }

    /**
     * Get fallback response
     */
    private function getFallbackResponse(string $message): array
    {
        $responses = [
            "I'm sorry, I'm having trouble processing your request right now. Please try again or contact our support team at (555) 123-4567.",
            "I apologize for the technical difficulty. For immediate assistance, please call our main hospital line or speak with a staff member.",
            "I'm experiencing some technical issues. Please contact our patient services department for assistance with your inquiry.",
        ];

        return [
            'response' => $responses[array_rand($responses)],
            'ai_generated' => false,
            'confidence' => 'low'
        ];
    }

    /**
     * Categorize patient query
     */
    public function categorizeQuery(string $message): string
    {
        $categories = [
            'appointment' => ['appointment', 'schedule', 'booking', 'visit', 'meeting'],
            'prescription' => ['prescription', 'medication', 'drug', 'medicine', 'refill'],
            'billing' => ['bill', 'payment', 'insurance', 'cost', 'charge'],
            'medical' => ['symptom', 'pain', 'treatment', 'diagnosis', 'condition'],
            'general' => ['hours', 'location', 'contact', 'information', 'help'],
        ];

        $message = strtolower($message);
        
        foreach ($categories as $category => $keywords) {
            foreach ($keywords as $keyword) {
                if (strpos($message, $keyword) !== false) {
                    return $category;
                }
            }
        }

        return 'general';
    }

    /**
     * Get contextual suggestions based on query category
     */
    public function getSuggestions(string $category, array $context = []): array
    {
        $suggestions = [
            'appointment' => [
                'Schedule a new appointment',
                'Reschedule existing appointment',
                'Cancel appointment',
                'View appointment details'
            ],
            'prescription' => [
                'Request prescription refill',
                'Check prescription status',
                'View current medications',
                'Report medication issues'
            ],
            'billing' => [
                'View current bills',
                'Make payment',
                'Insurance information',
                'Payment history'
            ],
            'medical' => [
                'Schedule consultation',
                'View medical records',
                'Emergency contact',
                'Urgent care information'
            ],
            'general' => [
                'Hospital hours',
                'Contact information',
                'Location directions',
                'General information'
            ]
        ];

        return $suggestions[$category] ?? $suggestions['general'];
    }

    /**
     * Generate appointment scheduling response
     */
    public function handleAppointmentRequest(string $message, array $patientInfo): array
    {
        $prompt = "The patient is requesting help with appointments. ";
        $prompt .= "Patient: " . ($patientInfo['name'] ?? 'Unknown') . "\n";
        $prompt .= "Message: \"{$message}\"\n\n";
        $prompt .= "Provide helpful guidance on appointment scheduling, including available options and next steps.";

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(15)->post($this->baseUrl, [
                'model' => 'gpt-3.5-turbo',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a helpful hospital assistant specializing in appointment scheduling. Be friendly and provide clear next steps.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 300,
                'temperature' => 0.6,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return [
                    'response' => $result['choices'][0]['message']['content'],
                    'ai_generated' => true,
                    'suggestions' => $this->getSuggestions('appointment', $patientInfo)
                ];
            }
        } catch (\Exception $e) {
            \Log::error('AI Appointment Response Failed: ' . $e->getMessage());
        }

        return [
            'response' => 'I can help you with appointment scheduling. You can schedule a new appointment, reschedule an existing one, or view your upcoming appointments. Would you like me to guide you through any of these options?',
            'ai_generated' => false,
            'suggestions' => $this->getSuggestions('appointment', $patientInfo)
        ];
    }

    /**
     * Generate prescription-related response
     */
    public function handlePrescriptionRequest(string $message, array $patientInfo): array
    {
        $prompt = "The patient is asking about prescriptions. ";
        $prompt .= "Patient: " . ($patientInfo['name'] ?? 'Unknown') . "\n";
        $prompt .= "Message: \"{$message}\"\n\n";
        $prompt .= "Provide helpful information about prescription management, refills, and related services.";

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(15)->post($this->baseUrl, [
                'model' => 'gpt-3.5-turbo',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => 'You are a helpful hospital assistant specializing in prescription management. Provide clear guidance on medication-related services.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $prompt
                    ]
                ],
                'max_tokens' => 300,
                'temperature' => 0.6,
            ]);

            if ($response->successful()) {
                $result = $response->json();
                return [
                    'response' => $result['choices'][0]['message']['content'],
                    'ai_generated' => true,
                    'suggestions' => $this->getSuggestions('prescription', $patientInfo)
                ];
            }
        } catch (\Exception $e) {
            \Log::error('AI Prescription Response Failed: ' . $e->getMessage());
        }

        return [
            'response' => 'I can help you with prescription-related questions. You can request refills, check prescription status, or get information about your medications. What would you like to know?',
            'ai_generated' => false,
            'suggestions' => $this->getSuggestions('prescription', $patientInfo)
        ];
    }
}