<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Visit;

class VisitPolicy
{
    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        // All authenticated users can view visits
        return true;
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Visit $visit): bool
    {
        // All authenticated users can view visit details
        return true;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        // Only front desk and doctors can check in patients
        return $user->isFrontDesk() || $user->isDoctor();
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Visit $visit): bool
    {
        // Only front desk and doctors can update visit details
        return $user->isFrontDesk() || $user->isDoctor();
    }

    /**
     * Determine whether the user can update visit status.
     */
    public function updateStatus(User $user, Visit $visit, string $status): bool
    {
        // Status-specific permissions
        return match($status) {
            Visit::STATUS_CHECKED_IN => $user->isFrontDesk() || $user->isDoctor(),
            Visit::STATUS_WAITING_DOCTOR => $user->isNurse(),
            Visit::STATUS_IN_CONSULTATION => $user->isDoctor(),
            Visit::STATUS_LAB_REQUESTED => $user->isDoctor(),
            Visit::STATUS_PRESCRIPTION_ISSUED => $user->isDoctor(),
            Visit::STATUS_PHARMACY_DISPENSED => $user->isPharmacy() || $user->isDoctor(),
            Visit::STATUS_BILLING_PENDING => $user->canManageBills(),
            Visit::STATUS_CHECKED_OUT => $user->canManageBills(),
            default => false,
        };
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Visit $visit): bool
    {
        // Only doctors can delete visits
        return $user->isDoctor();
    }
}
