<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Visit extends Model
{
    use HasFactory;

    protected $fillable = [
        'visit_number',
        'patient_id',
        'created_by',
        'status',
        'checked_in_at',
        'checked_out_at',
        'chief_complaint',
        'visit_notes',
    ];

    protected $casts = [
        'checked_in_at' => 'datetime',
        'checked_out_at' => 'datetime',
    ];

    // Status constants
    const STATUS_CHECKED_IN = 'checked_in';
    const STATUS_WAITING_NURSE = 'waiting_nurse';
    const STATUS_WAITING_DOCTOR = 'waiting_doctor';
    const STATUS_IN_CONSULTATION = 'in_consultation';
    const STATUS_LAB_REQUESTED = 'lab_requested';
    const STATUS_PRESCRIPTION_ISSUED = 'prescription_issued';
    const STATUS_PHARMACY_DISPENSED = 'pharmacy_dispensed';
    const STATUS_BILLING_PENDING = 'billing_pending';
    const STATUS_ADMITTED = 'admitted';
    const STATUS_CHECKED_OUT = 'checked_out';

    /**
     * Boot the model and generate visit number
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($visit) {
            if (empty($visit->visit_number)) {
                $visit->visit_number = static::generateVisitNumber();
            }
            if (empty($visit->checked_in_at)) {
                $visit->checked_in_at = now();
            }
        });
    }

    /**
     * Generate unique visit number
     */
    public static function generateVisitNumber(): string
    {
        $year = now()->format('Y');
        $lastVisit = static::whereYear('created_at', $year)
            ->orderBy('id', 'desc')
            ->first();
        
        $number = $lastVisit ? ((int) substr($lastVisit->visit_number, -6)) + 1 : 1;
        
        return 'VIS-' . $year . '-' . str_pad($number, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Relationships
     */
    public function patient(): BelongsTo
    {
        return $this->belongsTo(Patient::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function medicalRecords(): HasMany
    {
        return $this->hasMany(MedicalRecord::class);
    }

    public function consultations(): HasMany
    {
        return $this->hasMany(Consultation::class);
    }

    public function prescriptions(): HasMany
    {
        return $this->hasMany(Prescription::class);
    }

    public function bills(): HasMany
    {
        return $this->hasMany(Bill::class);
    }

    public function labRequests(): HasMany
    {
        return $this->hasMany(LabRequest::class);
    }

    public function admission(): HasOne
    {
        return $this->hasOne(Admission::class, 'visit_id');
    }

    /**
     * Check if visit is active (not checked out)
     */
    public function isActive(): bool
    {
        return $this->status !== self::STATUS_CHECKED_OUT && $this->checked_out_at === null;
    }

    /**
     * Update visit status
     */
    public function updateStatus(string $status): bool
    {
        $validStatuses = [
            self::STATUS_CHECKED_IN,
            self::STATUS_WAITING_NURSE,
            self::STATUS_WAITING_DOCTOR,
            self::STATUS_IN_CONSULTATION,
            self::STATUS_LAB_REQUESTED,
            self::STATUS_PRESCRIPTION_ISSUED,
            self::STATUS_PHARMACY_DISPENSED,
            self::STATUS_BILLING_PENDING,
            self::STATUS_ADMITTED,
            self::STATUS_CHECKED_OUT,
        ];

        if (!in_array($status, $validStatuses)) {
            return false;
        }

        $this->status = $status;

        if ($status === self::STATUS_CHECKED_OUT) {
            $this->checked_out_at = now();
        }

        return $this->save();
    }

    /**
     * Get status badge color
     */
    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            self::STATUS_CHECKED_IN => 'blue',
            self::STATUS_WAITING_NURSE => 'yellow',
            self::STATUS_WAITING_DOCTOR => 'orange',
            self::STATUS_IN_CONSULTATION => 'purple',
            self::STATUS_LAB_REQUESTED => 'indigo',
            self::STATUS_PRESCRIPTION_ISSUED => 'pink',
            self::STATUS_PHARMACY_DISPENSED => 'cyan',
            self::STATUS_BILLING_PENDING => 'amber',
            self::STATUS_ADMITTED => 'red',
            self::STATUS_CHECKED_OUT => 'green',
            default => 'gray',
        };
    }

    /**
     * Get human-readable status
     */
    public function getStatusLabelAttribute(): string
    {
        return match($this->status) {
            self::STATUS_CHECKED_IN => 'Checked In',
            self::STATUS_WAITING_NURSE => 'Waiting for Nurse',
            self::STATUS_WAITING_DOCTOR => 'Waiting for Doctor',
            self::STATUS_IN_CONSULTATION => 'In Consultation',
            self::STATUS_LAB_REQUESTED => 'Lab Requested',
            self::STATUS_PRESCRIPTION_ISSUED => 'Prescription Issued',
            self::STATUS_PHARMACY_DISPENSED => 'Pharmacy Dispensed',
            self::STATUS_BILLING_PENDING => 'Billing Pending',
            self::STATUS_ADMITTED => 'Admitted',
            self::STATUS_CHECKED_OUT => 'Checked Out',
            default => 'Unknown',
        };
    }
}
