<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Consultation extends Model
{
    use HasFactory;

    protected $fillable = [
        'patient_id',
        'visit_id',
        'doctor_id',
        'consultation_type',
        'chief_complaint',
        'history_of_present_illness',
        'physical_examination',
        'assessment',
        'plan',
        'diagnosis',
        'treatment_plan',
        'follow_up_instructions',
        'consultation_date',
        'consultation_time',
        'duration',
        'status',
        'notes',
        'vital_signs',
        'referrals',
        'prescriptions_given',
    ];

    protected $casts = [
        'consultation_date' => 'date',
        'consultation_time' => 'datetime',
        'vital_signs' => 'array',
        'referrals' => 'array',
        'prescriptions_given' => 'array',
    ];

    public function patient(): BelongsTo
    {
        return $this->belongsTo(Patient::class);
    }

    public function doctor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'doctor_id');
    }

    public function visit(): BelongsTo
    {
        return $this->belongsTo(Visit::class);
    }

    public function prescriptions(): HasMany
    {
        return $this->hasMany(Prescription::class);
    }

    public function getStatusBadgeColorAttribute(): string
    {
        return match($this->status) {
            'completed' => 'green',
            'in_progress' => 'blue',
            'cancelled' => 'red',
            'scheduled' => 'yellow',
            default => 'gray'
        };
    }

    public function getDurationFormattedAttribute(): string
    {
        $hours = floor($this->duration / 60);
        $minutes = $this->duration % 60;
        
        if ($hours > 0) {
            return "{$hours}h {$minutes}m";
        }
        
        return "{$minutes}m";
    }
}