<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Appointment extends Model
{
    use HasFactory;

    protected $fillable = [
        'patient_id',
        'doctor_id',
        'appointment_date',
        'appointment_time',
        'duration',
        'type',
        'status',
        'notes',
        'reason',
        'guest_name',
        'guest_phone',
        'guest_email',
        'checked_in_at',
    ];

    protected $casts = [
        'appointment_date' => 'date',
        'appointment_time' => 'string',
        'duration' => 'integer',
        'checked_in_at' => 'datetime',
    ];

    const STATUS_SCHEDULED = 'scheduled';
    const STATUS_CONFIRMED = 'confirmed';
    const STATUS_IN_PROGRESS = 'in_progress';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';
    const STATUS_NO_SHOW = 'no_show';

    const TYPE_CONSULTATION = 'consultation';
    const TYPE_FOLLOW_UP = 'follow_up';
    const TYPE_CHECKUP = 'checkup';
    const TYPE_EMERGENCY = 'emergency';

    public function patient(): BelongsTo
    {
        return $this->belongsTo(Patient::class);
    }

    public function doctor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'doctor_id');
    }

    public function getDateTimeAttribute(): string
    {
        return $this->appointment_date->format('Y-m-d') . ' ' . $this->appointment_time;
    }

    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            self::STATUS_SCHEDULED => 'yellow',
            self::STATUS_CONFIRMED => 'blue',
            self::STATUS_IN_PROGRESS => 'purple',
            self::STATUS_COMPLETED => 'green',
            self::STATUS_CANCELLED => 'red',
            self::STATUS_NO_SHOW => 'gray',
            default => 'gray',
        };
    }

    /**
     * Check if this is a guest/walk-in appointment
     */
    public function isGuest(): bool
    {
        return $this->patient_id === null && !empty($this->guest_name);
    }

    /**
     * Get patient name (regular or guest)
     */
    public function getPatientNameAttribute(): string
    {
        // Check if this is a guest appointment
        if ($this->patient_id === null) {
            return $this->guest_name ?? 'Guest Patient';
        }
        
        // For regular patients, check if relationship is loaded
        if ($this->relationLoaded('patient') && $this->patient) {
            return $this->patient->full_name;
        }
        
        // If relationship not loaded, try to access it
        try {
            return $this->patient ? $this->patient->full_name : 'N/A';
        } catch (\Exception $e) {
            return 'N/A';
        }
    }

    /**
     * Check in the appointment
     */
    public function checkIn(): void
    {
        $this->update([
            'status' => self::STATUS_CONFIRMED,
            'checked_in_at' => now(),
        ]);
    }
}

