<?php

namespace App\Livewire\Services;

use App\Models\Service;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.app')]
class ServiceManagement extends Component
{
    use WithPagination;

    public $search = '';
    public $categoryFilter = '';
    public $statusFilter = '';
    
    // Form fields
    public $showModal = false;
    public $editingService = null;
    public $code = '';
    public $name = '';
    public $category = '';
    public $subcategory = '';
    public $service_key = '';
    public $price = 0;
    public $unit = 'per_unit';
    public $description = '';
    public $is_active = true;
    public $sort_order = 0;

    protected $queryString = [
        'search' => ['except' => ''],
        'categoryFilter' => ['except' => ''],
        'statusFilter' => ['except' => ''],
    ];

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Only CMD and developer can access
        $user = auth()->user();
        if (!$user->isDeveloper() && !$user->isCmd()) {
            abort(403, 'Access denied. This page is for CMD and developers only.');
        }
    }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->showModal = true;
    }

    public function openEditModal($serviceId)
    {
        $service = Service::findOrFail($serviceId);
        $this->editingService = $service;
        $this->code = $service->code;
        $this->name = $service->name;
        $this->category = $service->category;
        $this->subcategory = $service->subcategory ?? '';
        $this->service_key = $service->service_key ?? '';
        $this->price = $service->price;
        $this->unit = $service->unit;
        $this->description = $service->description ?? '';
        $this->is_active = $service->is_active;
        $this->sort_order = $service->sort_order;
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->editingService = null;
        $this->code = '';
        $this->name = '';
        $this->category = '';
        $this->subcategory = '';
        $this->service_key = '';
        $this->price = 0;
        $this->unit = 'per_unit';
        $this->description = '';
        $this->is_active = true;
        $this->sort_order = 0;
        $this->resetErrorBag();
    }

    public function save()
    {
        $rules = [
            'code' => 'required|string|max:255|unique:services,code' . ($this->editingService ? ',' . $this->editingService->id : ''),
            'name' => 'required|string|max:255',
            'category' => 'required|in:consultation,lab_test,prescription,other',
            'price' => 'required|numeric|min:0',
            'unit' => 'required|in:per_unit,per_quantity,per_hour,per_day',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ];

        $this->validate($rules);

        $data = [
            'code' => strtoupper($this->code),
            'name' => ucwords(strtolower($this->name)),
            'category' => $this->category,
            'subcategory' => $this->subcategory ?: null,
            'service_key' => $this->service_key ?: null,
            'price' => $this->price,
            'unit' => $this->unit,
            'description' => $this->description ?: null,
            'is_active' => $this->is_active,
            'sort_order' => $this->sort_order,
        ];

        if ($this->editingService) {
            $this->editingService->update($data);
            session()->flash('message', 'Service updated successfully!');
        } else {
            Service::create($data);
            session()->flash('message', 'Service created successfully!');
        }

        $this->closeModal();
    }

    public function delete($serviceId)
    {
        $service = Service::findOrFail($serviceId);
        $serviceName = $service->name;
        $service->delete();
        
        session()->flash('message', "Service '{$serviceName}' deleted successfully!");
    }

    public function toggleStatus($serviceId)
    {
        $service = Service::findOrFail($serviceId);
        $service->update(['is_active' => !$service->is_active]);
        
        $status = $service->is_active ? 'activated' : 'deactivated';
        session()->flash('message', "Service '{$service->name}' {$status} successfully!");
    }

    public function render()
    {
        $services = Service::query()
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('code', 'like', '%' . $this->search . '%')
                      ->orWhere('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->categoryFilter, function ($query) {
                $query->where('category', $this->categoryFilter);
            })
            ->when($this->statusFilter === 'active', function ($query) {
                $query->where('is_active', true);
            })
            ->when($this->statusFilter === 'inactive', function ($query) {
                $query->where('is_active', false);
            })
            ->orderBy('category')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->paginate(15);

        $categories = Service::distinct()->pluck('category')->sort()->values();

        return view('livewire.services.service-management', [
            'services' => $services,
            'categories' => $categories,
        ]);
    }
}
