<?php

namespace App\Livewire\Pharmacy;

use App\Models\Prescription;
use App\Models\Patient;
use App\Models\Drug;
use App\Models\User;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Carbon\Carbon;
use Illuminate\Support\Str;

#[Layout('layouts.app')]
class PrescriptionList extends Component
{
    use WithPagination;

    public $search = '';
    public $statusFilter = '';
    public $showCreateModal = false;
    public $showEditModal = false;
    public $showViewModal = false;
    public $editingPrescription = null;
    public $selectedPrescription = null;

    // Form fields
    public $patient_id = '';
    public $doctor_id = '';
    public $drug_id = '';
    public $dosage = '';
    public $frequency = '';
    public $duration = 7;
    public $quantity = '';
    public $instructions = '';
    public $prescription_date = '';
    public $valid_until = '';
    public $status = 'active';
    public $notes = '';
    public $max_refills = 0;
    
    // Patient search
    public $patientSearch = '';
    public $searchResults = [];

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => ''],
    ];

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }
    }

    protected function rules()
    {
        return [
            'patient_id' => 'required|exists:patients,id',
            'doctor_id' => 'required|exists:users,id',
            'drug_id' => 'required|exists:drugs,id',
            'dosage' => 'required|string|max:255',
            'frequency' => 'required|string|max:255',
            'duration' => 'required|integer|min:1|max:365',
            'quantity' => 'required|integer|min:1',
            'instructions' => 'nullable|string',
            'prescription_date' => 'required|date',
            'valid_until' => 'required|date|after:prescription_date',
            'status' => 'required|in:active,dispensed,expired,cancelled',
            'notes' => 'nullable|string',
            'max_refills' => 'required|integer|min:0|max:12',
        ];
    }


    public $visit_id = ''; // Optional visit ID

    public function openCreateModal($visitId = null)
    {
        $this->resetForm();
        $this->doctor_id = auth()->id();
        $this->prescription_date = now()->format('Y-m-d');
        $this->valid_until = now()->addDays(30)->format('Y-m-d');
        $this->visit_id = $visitId ?? '';
        $this->showCreateModal = true;
    }

    public function openViewModal($prescriptionId)
    {
        $this->selectedPrescription = Prescription::with(['patient', 'doctor', 'drug', 'visit'])->findOrFail($prescriptionId);
        $this->showViewModal = true;
    }

    public function openEditModal($prescriptionId)
    {
        // Close view modal if open
        $this->showViewModal = false;
        $this->selectedPrescription = null;
        
        $prescription = Prescription::findOrFail($prescriptionId);
        $this->editingPrescription = $prescription;
        $this->patient_id = $prescription->patient_id;
        if ($this->patient_id) {
            $patient = Patient::find($this->patient_id);
            if ($patient) {
                $this->patientSearch = $patient->full_name . ' (' . $patient->patient_id . ')';
            }
        }
        $this->doctor_id = $prescription->doctor_id;
        $this->drug_id = $prescription->drug_id;
        $this->dosage = $prescription->dosage;
        $this->frequency = $prescription->frequency;
        $this->duration = $prescription->duration;
        $this->quantity = $prescription->quantity;
        $this->instructions = $prescription->instructions;
        $this->prescription_date = $prescription->prescription_date->format('Y-m-d');
        $this->valid_until = $prescription->valid_until->format('Y-m-d');
        $this->status = $prescription->status;
        $this->notes = $prescription->notes;
        $this->max_refills = $prescription->max_refills;
        $this->showEditModal = true;
    }

    public function closeModals()
    {
        $this->showCreateModal = false;
        $this->showEditModal = false;
        $this->showViewModal = false;
        $this->editingPrescription = null;
        $this->selectedPrescription = null;
        $this->resetForm();
    }

    public function closeViewModal()
    {
        $this->showViewModal = false;
        $this->selectedPrescription = null;
    }

    public function resetForm()
    {
        $this->patient_id = '';
        $this->visit_id = '';
        $this->doctor_id = '';
        $this->drug_id = '';
        $this->dosage = '';
        $this->frequency = '';
        $this->duration = 7;
        $this->quantity = '';
        $this->instructions = '';
        $this->prescription_date = '';
        $this->valid_until = '';
        $this->status = 'active';
        $this->notes = '';
        $this->max_refills = 0;
        $this->patientSearch = '';
        $this->searchResults = [];
    }
    
    public function searchPatients()
    {
        if (strlen($this->patientSearch) < 2) {
            $this->searchResults = [];
            return;
        }

        $searchTerm = strtolower($this->patientSearch);
        $this->searchResults = Patient::query()
            ->where(function ($query) use ($searchTerm) {
                $query->whereRaw('LOWER(first_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(last_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(patient_id) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(phone) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(email) like ?', ['%' . $searchTerm . '%']);
            })
            ->limit(10)
            ->get();
    }

    public function selectPatient($patientId)
    {
        $this->patient_id = $patientId;
        $patient = Patient::find($patientId);
        if ($patient) {
            $this->patientSearch = $patient->full_name . ' (' . $patient->patient_id . ')';
        }
        $this->searchResults = [];
    }

    public function save()
    {
        // Nurses cannot create or edit prescriptions
        if (auth()->user()->isNurse()) {
            session()->flash('error', 'Nurses cannot create or edit prescriptions.');
            return;
        }

        $this->validate();

        if ($this->editingPrescription) {
            $this->editingPrescription->update([
                'patient_id' => $this->patient_id,
                'doctor_id' => $this->doctor_id,
                'drug_id' => $this->drug_id,
                'dosage' => $this->dosage,
                'frequency' => $this->frequency,
                'duration' => $this->duration,
                'quantity' => $this->quantity,
                'instructions' => $this->instructions,
                'prescription_date' => $this->prescription_date,
                'valid_until' => $this->valid_until,
                'status' => $this->status,
                'notes' => $this->notes,
                'max_refills' => $this->max_refills,
            ]);
            session()->flash('message', 'Prescription updated successfully!');
        } else {
            $prescriptionData = [
                'patient_id' => $this->patient_id,
                'visit_id' => $this->visit_id ?: null,
                'doctor_id' => $this->doctor_id,
                'drug_id' => $this->drug_id,
                'prescription_number' => 'RX-' . strtoupper(Str::random(8)),
                'dosage' => $this->dosage,
                'frequency' => $this->frequency,
                'duration' => $this->duration,
                'quantity' => $this->quantity,
                'instructions' => $this->instructions,
                'prescription_date' => $this->prescription_date,
                'valid_until' => $this->valid_until,
                'status' => $this->status,
                'notes' => $this->notes,
                'refill_count' => 0,
                'max_refills' => $this->max_refills,
            ];
            Prescription::create($prescriptionData);
            
            // Update visit status if visit_id is provided
            if ($this->visit_id) {
                $visit = \App\Models\Visit::find($this->visit_id);
                if ($visit && $visit->status !== \App\Models\Visit::STATUS_PRESCRIPTION_ISSUED) {
                    $visit->updateStatus(\App\Models\Visit::STATUS_PRESCRIPTION_ISSUED);
                }
            }
            
            session()->flash('message', 'Prescription created successfully!');
        }

        $this->closeModals();
    }

    public function delete($prescriptionId)
    {
        // Nurses cannot delete prescriptions
        if (auth()->user()->isNurse()) {
            session()->flash('error', 'Nurses cannot delete prescriptions.');
            return;
        }

        Prescription::findOrFail($prescriptionId)->delete();
        session()->flash('message', 'Prescription deleted successfully!');
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function render()
    {
        $prescriptions = Prescription::query()
            ->with(['patient', 'doctor', 'drug'])
            ->when($this->search, function ($query) {
                $searchTerm = strtolower($this->search);
                $query->whereHas('patient', function ($q) use ($searchTerm) {
                    $q->whereRaw('LOWER(first_name) LIKE ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(last_name) LIKE ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(patient_id) LIKE ?', ['%' . $searchTerm . '%']);
                });
            })
            ->when($this->statusFilter, function ($query) {
                $query->where('status', $this->statusFilter);
            })
            ->when(!$this->search && !$this->statusFilter, function ($query) {
                // Show only today's prescriptions by default when no filters are applied
                $query->whereDate('prescription_date', today());
            })
            ->orderBy('prescription_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        $patients = Patient::orderBy('first_name')->get();
        $drugs = Drug::where('is_active', true)->orderBy('name')->get();
        // Only show doctors (users with role 'doctor', 'staff', or null for backward compatibility)
        $doctors = User::where(function($query) {
            $query->where('role', 'doctor')
                  ->orWhere('role', 'staff')
                  ->orWhereNull('role');
        })->orderBy('name')->get();

        return view('livewire.pharmacy.prescription-list', [
            'prescriptions' => $prescriptions,
            'patients' => $patients,
            'drugs' => $drugs,
            'doctors' => $doctors,
        ]);
    }
}

