<?php

namespace App\Livewire\Patients;

use App\Models\Patient;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

#[Layout('layouts.app')]
class PatientRegistration extends Component
{
    use WithFileUploads;

    // Patient fields - only vital details
    public $first_name = '';
    public $last_name = '';
    public $date_of_birth = '';
    public $gender = '';
    public $email = '';
    public $phone = '';

    protected function rules()
    {
        return [
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'date_of_birth' => ['required', 'date', 'before:today'],
            'gender' => ['required', 'in:male,female,other'],
            'email' => ['nullable', 'string', 'email', 'max:255'],
            'phone' => ['nullable', 'string', 'max:20'],
        ];
    }

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Doctors cannot create patients
        if (auth()->user()->isDoctor()) {
            abort(403, 'Access denied. Doctors cannot create patients.');
        }
    }

    public function register()
    {
        // Doctors cannot create patients
        if (auth()->user()->isDoctor()) {
            session()->flash('error', 'Doctors cannot create patients.');
            return redirect()->route('patients.index');
        }

        $this->validate();

        // Generate unique patient ID
        $patientId = 'P' . strtoupper(Str::random(6));

        // Create patient record - track which staff member registered the patient
        $patient = Patient::create([
            'user_id' => auth()->id(), // Track which staff member created this patient
            'patient_id' => $patientId,
            'first_name' => $this->first_name,
            'last_name' => $this->last_name,
            'date_of_birth' => $this->date_of_birth,
            'gender' => $this->gender,
            'email' => $this->email ?: null,
            'phone' => $this->phone ?: null,
            // Other fields can be filled later via patient profile
            'address' => null,
            'emergency_contact_name' => null,
            'emergency_contact_phone' => null,
            'insurance_provider' => null,
            'insurance_number' => null,
            'medical_history' => null,
            'allergies' => null,
            'blood_type' => null,
            'height' => null,
            'weight' => null,
        ]);

        session()->flash('message', 'Patient registered successfully! Patient ID: ' . $patientId);

        // Reset form
        $this->reset();

        return redirect()->route('patients.index');
    }

    public function render()
    {
        return view('livewire.patients.patient-registration');
    }
}