<?php

namespace App\Livewire\Patients;

use App\Models\Patient;
use App\Models\Visit;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Str;

#[Layout('layouts.app')]
class PatientList extends Component
{
    use WithPagination;

    public $search = '';
    public $sortBy = 'created_at';
    public $sortDirection = 'desc';
    public $showCreateModal = false;
    public $showCreateVisitModal = false;
    public $selectedPatientForVisit = null;
    public $chiefComplaint = '';
    public $visitNotes = '';

    // Patient registration fields
    public $first_name = '';
    public $last_name = '';
    public $date_of_birth = '';
    public $gender = '';
    public $email = '';
    public $phone = '';

    protected $queryString = [
        'search' => ['except' => ''],
        'sortBy' => ['except' => 'created_at'],
        'sortDirection' => ['except' => 'desc'],
    ];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->showCreateModal = true;
    }

    public function closeCreateModal()
    {
        $this->showCreateModal = false;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->first_name = '';
        $this->last_name = '';
        $this->date_of_birth = '';
        $this->gender = '';
        $this->email = '';
        $this->phone = '';
        $this->resetErrorBag();
    }

    protected function rules()
    {
        return [
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'date_of_birth' => ['required', 'date', 'before:today'],
            'gender' => ['required', 'in:male,female,other'],
            'email' => ['nullable', 'string', 'email', 'max:255'],
            'phone' => ['nullable', 'string', 'max:20'],
        ];
    }

    public function register()
    {
        // Doctors cannot create patients
        if (auth()->user()->isDoctor()) {
            session()->flash('error', 'Doctors cannot create patients.');
            return;
        }

        $this->validate();

        // Generate unique patient ID
        $patientId = 'P' . strtoupper(Str::random(6));

        // Create patient record - track which staff member registered the patient
        $patient = Patient::create([
            'user_id' => auth()->id(), // Track which staff member created this patient
            'patient_id' => $patientId,
            'first_name' => $this->first_name,
            'last_name' => $this->last_name,
            'date_of_birth' => $this->date_of_birth,
            'gender' => $this->gender,
            'email' => $this->email ?: null,
            'phone' => $this->phone ?: null,
            // Other fields can be filled later via patient profile
            'address' => null,
            'emergency_contact_name' => null,
            'emergency_contact_phone' => null,
            'insurance_provider' => null,
            'insurance_number' => null,
            'medical_history' => null,
            'allergies' => null,
            'blood_type' => null,
            'height' => null,
            'weight' => null,
        ]);

        // If there's a search term, check if the new patient matches
        // If it matches, it will automatically appear due to live search
        // Reset pagination to first page to show the new patient
        $this->resetPage();
        
        // Close modal and reset form
        $this->closeCreateModal();
        
        // Show success message
        session()->flash('message', 'Patient registered successfully! Patient ID: ' . $patientId);
    }

    public function openCreateVisitModal($patientId)
    {
        $patient = Patient::findOrFail($patientId);
        
        // Check if patient already has an active visit
        if ($patient->active_visit) {
            session()->flash('error', 'Patient already has an active visit: ' . $patient->active_visit->visit_number . '. Cannot create another visit.');
            return;
        }
        
        $this->selectedPatientForVisit = $patient;
        $this->chiefComplaint = '';
        $this->visitNotes = '';
        $this->showCreateVisitModal = true;
    }

    public function closeCreateVisitModal()
    {
        $this->showCreateVisitModal = false;
        $this->selectedPatientForVisit = null;
        $this->chiefComplaint = '';
        $this->visitNotes = '';
        $this->resetErrorBag();
    }

    public function createVisit()
    {
        $rules = [
            'visitNotes' => 'nullable|string|max:2000',
        ];
        
        // Doctors, developers, and cmd can enter chief complaint
        if (auth()->user()->canEnterChiefComplaint()) {
            $rules['chiefComplaint'] = 'nullable|string|max:1000';
        }
        
        $this->validate($rules, [], [
            'chiefComplaint' => 'chief complaint',
            'visitNotes' => 'visit notes',
        ]);

        if (!$this->selectedPatientForVisit) {
            session()->flash('error', 'Please select a patient.');
            return;
        }

        // Check if patient already has an active visit
        $activeVisit = $this->selectedPatientForVisit->active_visit;

        if ($activeVisit) {
            session()->flash('error', 'Patient already has an active visit: ' . $activeVisit->visit_number);
            $this->closeCreateVisitModal();
            return;
        }

        // Create new visit and send to nurse
        // Doctors, developers, and cmd can enter chief complaint
        $visit = Visit::create([
            'patient_id' => $this->selectedPatientForVisit->id,
            'created_by' => auth()->id(),
            'status' => Visit::STATUS_WAITING_NURSE, // Immediately send to nursing station
            'chief_complaint' => auth()->user()->canEnterChiefComplaint() ? ($this->chiefComplaint ?: null) : null,
            'visit_notes' => $this->visitNotes ?: null,
        ]);

        session()->flash('message', 'Visit created successfully! Visit Number: ' . $visit->visit_number . ' - Patient sent to nursing station.');
        $this->closeCreateVisitModal();
        
        // Refresh the component to update button states
        $this->dispatch('visit-created');
    }

    public function render()
    {
        $searchTerm = strtolower($this->search);
        
        $patients = Patient::query()
            ->with(['visits' => function ($query) {
                $query->where('status', '!=', Visit::STATUS_CHECKED_OUT)
                      ->whereNull('checked_out_at')
                      ->latest('checked_in_at');
            }])
            ->when($this->search, function ($query) use ($searchTerm) {
                $query->where(function ($q) use ($searchTerm) {
                    $q->whereRaw('LOWER(first_name) LIKE ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(last_name) LIKE ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(patient_id) LIKE ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(phone) LIKE ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(email) LIKE ?', ['%' . $searchTerm . '%']);
                });
            })
            ->orderBy($this->sortBy, $this->sortDirection)
            ->paginate(10);

        return view('livewire.patients.patient-list', [
            'patients' => $patients,
        ]);
    }
}
