<?php

namespace App\Livewire\Nursing;

use App\Models\Visit;
use App\Models\Patient;
use App\Models\MedicalRecord;
use App\Services\AuditLogService;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.app')]
class NursingVitals extends Component
{
    use WithPagination;

    public $selectedVisitId = null;
    public $selectedVisit = null;
    public $selectedPatient = null;
    
    // Vitals fields
    public $blood_pressure = '';
    public $temperature = '';
    public $pulse = '';
    public $height = '';
    public $weight = '';
    public $respiratory_rate = '';
    public $oxygen_saturation = '';
    
    // Medical history fields (longitudinal)
    public $medical_history = '';
    public $allergies = '';
    
    public $showVitalsModal = false;

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Check if user is a nurse, developer, or cmd
        $user = auth()->user();
        if (!$user->isNurse() && !$user->isDeveloper() && !$user->isCmd()) {
            abort(403, 'Access denied. This page is for nursing staff, developers, and CMD only.');
        }
    }

    public function selectVisit($visitId)
    {
        $this->selectedVisitId = $visitId;
        $this->selectedVisit = Visit::with(['patient'])->findOrFail($visitId);
        $this->selectedPatient = $this->selectedVisit->patient;
        
        // Load patient's current medical history and allergies
        $this->medical_history = $this->selectedPatient->medical_history ?? '';
        $this->allergies = $this->selectedPatient->allergies ?? '';
        
        // Load existing vitals if any
        $existingRecord = MedicalRecord::where('visit_id', $visitId)->first();
        if ($existingRecord && $existingRecord->vital_signs) {
            $vitals = $existingRecord->vital_signs;
            $this->blood_pressure = $vitals['blood_pressure'] ?? '';
            $this->temperature = $vitals['temperature'] ?? '';
            $this->pulse = $vitals['pulse'] ?? '';
            $this->height = $vitals['height'] ?? '';
            $this->weight = $vitals['weight'] ?? '';
            $this->respiratory_rate = $vitals['respiratory_rate'] ?? '';
            $this->oxygen_saturation = $vitals['oxygen_saturation'] ?? '';
        }
        
        $this->showVitalsModal = true;
    }

    public function closeVitalsModal()
    {
        $this->showVitalsModal = false;
        $this->resetVitalsForm();
    }

    public function resetVitalsForm()
    {
        $this->selectedVisitId = null;
        $this->selectedVisit = null;
        $this->selectedPatient = null;
        $this->blood_pressure = '';
        $this->temperature = '';
        $this->pulse = '';
        $this->height = '';
        $this->weight = '';
        $this->respiratory_rate = '';
        $this->oxygen_saturation = '';
        $this->medical_history = '';
        $this->allergies = '';
        $this->resetErrorBag();
    }

    public function saveVitals()
    {
        $this->validate([
            'blood_pressure' => 'nullable|string|max:20',
            'temperature' => 'nullable|numeric|min:90|max:110',
            'pulse' => 'nullable|integer|min:40|max:200',
            'height' => 'nullable|numeric|min:50|max:250',
            'weight' => 'nullable|numeric|min:10|max:500',
            'respiratory_rate' => 'nullable|integer|min:10|max:40',
            'oxygen_saturation' => 'nullable|integer|min:70|max:100',
            'medical_history' => 'nullable|string',
            'allergies' => 'nullable|string',
        ]);

        if (!$this->selectedVisit) {
            session()->flash('error', 'No visit selected.');
            return;
        }

        // Store old values for audit logging
        $oldMedicalHistory = $this->selectedPatient->medical_history;
        $oldAllergies = $this->selectedPatient->allergies;

        // Update patient's longitudinal medical history and allergies
        $this->selectedPatient->update([
            'medical_history' => $this->medical_history,
            'allergies' => $this->allergies,
            'height' => $this->height ?: $this->selectedPatient->height,
            'weight' => $this->weight ?: $this->selectedPatient->weight,
        ]);

        // Log changes with visit context
        if ($oldMedicalHistory !== $this->medical_history) {
            AuditLogService::logMedicalHistoryUpdate(
                $this->selectedPatient,
                $oldMedicalHistory,
                $this->medical_history,
                $this->selectedVisitId
            );
        }

        if ($oldAllergies !== $this->allergies) {
            AuditLogService::logAllergiesUpdate(
                $this->selectedPatient,
                $oldAllergies,
                $this->allergies,
                $this->selectedVisitId
            );
        }

        // Create or update medical record with vitals for this visit
        $vitalSigns = [
            'blood_pressure' => $this->blood_pressure,
            'temperature' => $this->temperature,
            'pulse' => $this->pulse,
            'height' => $this->height,
            'weight' => $this->weight,
            'respiratory_rate' => $this->respiratory_rate,
            'oxygen_saturation' => $this->oxygen_saturation,
        ];

        // Remove empty values
        $vitalSigns = array_filter($vitalSigns, function($value) {
            return $value !== '' && $value !== null;
        });

        $existingRecord = MedicalRecord::where('visit_id', $this->selectedVisitId)->first();
        
        if ($existingRecord) {
            // Update existing record
            $existingRecord->update([
                'vital_signs' => !empty($vitalSigns) ? $vitalSigns : null,
            ]);
        } else {
            // Create new medical record for this visit
            MedicalRecord::create([
                'patient_id' => $this->selectedPatient->id,
                'visit_id' => $this->selectedVisitId,
                'doctor_id' => auth()->id(), // Nurse ID
                'visit_date' => $this->selectedVisit->checked_in_at->toDateString(),
                'visit_type' => 'consultation',
                'vital_signs' => !empty($vitalSigns) ? $vitalSigns : null,
                'notes' => 'Vitals recorded by nursing staff',
            ]);
        }

        // Update visit status to waiting for doctor
        $this->selectedVisit->updateStatus(Visit::STATUS_WAITING_DOCTOR);

        session()->flash('message', 'Vitals recorded successfully! Patient is now waiting for doctor.');
        $this->closeVitalsModal();
        $this->resetPage();
    }

    public function render()
    {
        // Get visits waiting for nursing (checked_in or waiting_nurse status)
        $visits = Visit::with(['patient', 'createdBy'])
            ->whereIn('status', [Visit::STATUS_CHECKED_IN, Visit::STATUS_WAITING_NURSE])
            ->whereNull('checked_out_at')
            ->orderBy('checked_in_at', 'asc')
            ->paginate(15);

        return view('livewire.nursing.nursing-vitals', [
            'visits' => $visits,
        ]);
    }
}
