<?php

namespace App\Livewire\Consultations;

use App\Models\Consultation;
use App\Models\Patient;
use App\Models\User;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Carbon\Carbon;

#[Layout('layouts.app')]
class ConsultationList extends Component
{
    use WithPagination;

    public $search = '';
    public $statusFilter = '';
    public $showCreateModal = false;
    public $showEditModal = false;
    public $editingConsultation = null;

    public $patient_id = '';
    public $doctor_id = '';
    public $consultation_type = 'initial';
    public $chief_complaint = '';
    public $history_of_present_illness = '';
    public $physical_examination = '';
    public $diagnosis = '';
    public $treatment_plan = '';
    public $consultation_date = '';
    public $consultation_time = '';
    public $duration = 30;
    public $status = 'scheduled';
    public $notes = '';
    public $vital_signs = ['blood_pressure' => '', 'temperature' => '', 'heart_rate' => '', 'respiratory_rate' => '', 'oxygen_saturation' => ''];
    
    // Patient search
    public $patientSearch = '';
    public $searchResults = [];

    protected $queryString = ['search' => ['except' => ''], 'statusFilter' => ['except' => '']];

    public function mount()
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return $this->redirect(route('login'), navigate: true);
        }

        // Check if user can view consultations (nurses should not see consultations)
        $user = auth()->user();
        if (!$user->canViewConsultations()) {
            abort(403, 'Access denied. This page is not accessible to nurses.');
        }
    }

    public function updatingSearch() { $this->resetPage(); }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->doctor_id = auth()->id();
        $this->consultation_date = now()->format('Y-m-d');
        $this->consultation_time = now()->format('H:i');
        $this->showCreateModal = true;
    }

    public function openEditModal($id)
    {
        $consultation = Consultation::findOrFail($id);
        $this->editingConsultation = $consultation;
        $this->patient_id = $consultation->patient_id;
        if ($this->patient_id) {
            $patient = Patient::find($this->patient_id);
            if ($patient) {
                $this->patientSearch = $patient->full_name;
            }
        }
        $this->doctor_id = $consultation->doctor_id;
        $this->consultation_type = $consultation->consultation_type;
        $this->chief_complaint = $consultation->chief_complaint;
        $this->history_of_present_illness = $consultation->history_of_present_illness;
        $this->physical_examination = $consultation->physical_examination;
        $this->diagnosis = $consultation->diagnosis;
        $this->treatment_plan = $consultation->treatment_plan;
        $this->consultation_date = $consultation->consultation_date->format('Y-m-d');
        $this->consultation_time = $consultation->consultation_time->format('H:i');
        $this->duration = $consultation->duration;
        $this->status = $consultation->status;
        $this->notes = $consultation->notes;
        $this->vital_signs = $consultation->vital_signs ?? $this->vital_signs;
        $this->showEditModal = true;
    }

    public function closeModals()
    {
        $this->showCreateModal = false;
        $this->showEditModal = false;
        $this->editingConsultation = null;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->patient_id = '';
        $this->doctor_id = '';
        $this->consultation_type = 'initial';
        $this->chief_complaint = '';
        $this->history_of_present_illness = '';
        $this->physical_examination = '';
        $this->diagnosis = '';
        $this->treatment_plan = '';
        $this->consultation_date = '';
        $this->consultation_time = '';
        $this->duration = 30;
        $this->status = 'scheduled';
        $this->notes = '';
        $this->vital_signs = ['blood_pressure' => '', 'temperature' => '', 'heart_rate' => '', 'respiratory_rate' => '', 'oxygen_saturation' => ''];
        $this->patientSearch = '';
        $this->searchResults = [];
    }
    
    public function searchPatients()
    {
        if (strlen($this->patientSearch) < 2) {
            $this->searchResults = [];
            return;
        }

        $searchTerm = strtolower($this->patientSearch);
        $this->searchResults = Patient::query()
            ->where(function ($query) use ($searchTerm) {
                $query->whereRaw('LOWER(first_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(last_name) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(patient_id) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(phone) like ?', ['%' . $searchTerm . '%'])
                      ->orWhereRaw('LOWER(email) like ?', ['%' . $searchTerm . '%']);
            })
            ->limit(10)
            ->get();
    }

    public function selectPatient($patientId)
    {
        $this->patient_id = $patientId;
        $patient = Patient::find($patientId);
        if ($patient) {
            $this->patientSearch = $patient->full_name;
        }
        $this->searchResults = [];
    }

    public function save()
    {
        $this->validate([
            'patient_id' => 'required|exists:patients,id',
            'doctor_id' => 'required|exists:users,id',
            'consultation_type' => 'required|in:initial,follow_up,emergency,telemedicine,specialist',
            'chief_complaint' => 'required|string',
            'consultation_date' => 'required|date',
            'consultation_time' => 'required',
            'duration' => 'required|integer|min:15|max:480',
            'status' => 'required|in:scheduled,in_progress,completed,cancelled',
        ]);

        $consultationTime = Carbon::createFromFormat('Y-m-d H:i', $this->consultation_date . ' ' . $this->consultation_time);

        $data = [
            'patient_id' => $this->patient_id,
            'doctor_id' => $this->doctor_id,
            'consultation_type' => $this->consultation_type,
            'chief_complaint' => $this->chief_complaint,
            'history_of_present_illness' => $this->history_of_present_illness,
            'physical_examination' => $this->physical_examination,
            'diagnosis' => $this->diagnosis,
            'treatment_plan' => $this->treatment_plan,
            'consultation_date' => $this->consultation_date,
            'consultation_time' => $consultationTime,
            'duration' => $this->duration,
            'status' => $this->status,
            'notes' => $this->notes,
            'vital_signs' => $this->vital_signs,
        ];

        if ($this->editingConsultation) {
            $this->editingConsultation->update($data);
            session()->flash('message', 'Consultation updated successfully!');
        } else {
            Consultation::create($data);
            session()->flash('message', 'Consultation created successfully!');
        }

        $this->closeModals();
    }

    public function delete($id)
    {
        Consultation::findOrFail($id)->delete();
        session()->flash('message', 'Consultation deleted successfully!');
    }

    public function render()
    {
        $consultations = Consultation::with(['patient', 'doctor'])
            ->when($this->search, fn($q) => $q->whereHas('patient', fn($q) => $q->where('first_name', 'like', "%{$this->search}%")->orWhere('last_name', 'like', "%{$this->search}%")))
            ->when($this->statusFilter, fn($q) => $q->where('status', $this->statusFilter))
            ->orderBy('consultation_date', 'desc')
            ->paginate(15);

        return view('livewire.consultations.consultation-list', [
            'consultations' => $consultations,
            'patients' => Patient::orderBy('first_name')->get(),
            // Only show doctors (users with role 'doctor', 'staff', or null for backward compatibility)
            'doctors' => User::where(function($query) {
                $query->where('role', 'doctor')
                      ->orWhere('role', 'staff')
                      ->orWhereNull('role');
            })->orderBy('name')->get(),
        ]);
    }
}

