<?php

namespace App\Livewire\Admissions;

use App\Models\Admission;
use App\Models\Patient;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('layouts.app')]
class AdmissionList extends Component
{
    use WithPagination;

    public $search = '';
    public $statusFilter = 'admitted';
    public $showDischargeModal = false;
    public $selectedAdmission = null;
    public $discharge_notes = '';
    public $discharge_instructions = '';

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => 'admitted'],
    ];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function openDischargeModal($admissionId)
    {
        $this->selectedAdmission = Admission::with(['patient', 'visit'])->findOrFail($admissionId);
        $this->discharge_notes = '';
        $this->discharge_instructions = '';
        $this->showDischargeModal = true;
    }

    public function closeDischargeModal()
    {
        $this->showDischargeModal = false;
        $this->selectedAdmission = null;
        $this->discharge_notes = '';
        $this->discharge_instructions = '';
        $this->resetErrorBag();
    }

    public function discharge()
    {
        $this->validate([
            'discharge_notes' => 'nullable|string|max:2000',
            'discharge_instructions' => 'nullable|string|max:2000',
        ]);

        if (!$this->selectedAdmission) {
            session()->flash('error', 'No admission selected.');
            return;
        }

        $this->selectedAdmission->discharge(
            auth()->user(),
            $this->discharge_notes ?: null,
            $this->discharge_instructions ?: null
        );

        // Update visit status to checked_out if visit exists
        if ($this->selectedAdmission->visit) {
            $this->selectedAdmission->visit->updateStatus(\App\Models\Visit::STATUS_CHECKED_OUT);
        }

        session()->flash('message', 'Patient discharged successfully!');
        $this->closeDischargeModal();
        $this->resetPage();
    }

    public function render()
    {
        $query = Admission::with(['patient', 'visit', 'admittedBy', 'dischargedBy'])
            ->when($this->search, function ($q) {
                $q->whereHas('patient', function ($query) {
                    $query->where('first_name', 'like', '%' . $this->search . '%')
                          ->orWhere('last_name', 'like', '%' . $this->search . '%')
                          ->orWhere('patient_id', 'like', '%' . $this->search . '%');
                })->orWhere('admission_number', 'like', '%' . $this->search . '%')
                  ->orWhere('ward', 'like', '%' . $this->search . '%')
                  ->orWhere('room_number', 'like', '%' . $this->search . '%');
            })
            ->when($this->statusFilter, function ($q) {
                $q->where('status', $this->statusFilter);
            })
            ->orderBy('admission_date', 'desc');

        $admissions = $query->paginate(15);

        return view('livewire.admissions.admission-list', [
            'admissions' => $admissions,
        ]);
    }
}
