<?php

namespace App\Livewire\AI;

use App\Services\AIDrugInteractionService;
use App\Models\Drug;
use Livewire\Attributes\Layout;
use Livewire\Component;

#[Layout('layouts.app')]
class AIDrugInteractionChecker extends Component
{
    public $selectedDrugs = [];
    public $availableDrugs = [];
    public $interactionResults = null;
    public $isLoading = false;
    public $search = '';

    protected $rules = [
        'selectedDrugs' => 'required|array|min:2',
    ];

    protected $messages = [
        'selectedDrugs.required' => 'Please select at least 2 drugs to check for interactions.',
        'selectedDrugs.min' => 'Please select at least 2 drugs to check for interactions.',
    ];

    public function mount()
    {
        $this->availableDrugs = Drug::active()->get();
    }

    public function updatedSearch()
    {
        if (strlen($this->search) >= 2) {
            $this->availableDrugs = Drug::active()
                ->where(function ($query) {
                    $query->where('name', 'like', '%' . $this->search . '%')
                          ->orWhere('generic_name', 'like', '%' . $this->search . '%');
                })
                ->limit(10)
                ->get();
        } else {
            $this->availableDrugs = Drug::active()->limit(20)->get();
        }
    }

    public function addDrug($drugId)
    {
        if (!in_array($drugId, $this->selectedDrugs)) {
            $this->selectedDrugs[] = $drugId;
        }
    }

    public function removeDrug($drugId)
    {
        $this->selectedDrugs = array_filter($this->selectedDrugs, function ($id) use ($drugId) {
            return $id != $drugId;
        });
        $this->selectedDrugs = array_values($this->selectedDrugs);
    }

    public function checkInteractions()
    {
        $this->validate();
        
        $this->isLoading = true;
        $this->interactionResults = null;

        try {
            $aiService = new AIDrugInteractionService();
            $this->interactionResults = $aiService->checkInteractions($this->selectedDrugs);
        } catch (\Exception $e) {
            session()->flash('error', 'Failed to check drug interactions. Please try again.');
        } finally {
            $this->isLoading = false;
        }
    }

    public function getSelectedDrugsModelsProperty()
    {
        return Drug::whereIn('id', $this->selectedDrugs)->get();
    }

    public function getRiskLevelColorProperty()
    {
        if (!$this->interactionResults) {
            return 'gray';
        }

        return match($this->interactionResults['risk_level']) {
            'none' => 'green',
            'low' => 'yellow',
            'moderate' => 'orange',
            'high' => 'red',
            'severe' => 'red',
            default => 'gray'
        };
    }

    public function render()
    {
        return view('livewire.ai.a-i-drug-interaction-checker');
    }
}